extensions [sound]

breed [pigs pig]
breed [sounders sounder]

globals
[
  ; output, counters
  pig-count
  red-count            ; population of female fertile hogs
  yellow-count         ; population of female sterile hogs
  blue-count           ; population of male hogs
  adult-male-count
  juv-male-count
  female-count
  adult-female-count
  juv-female-count
  sounder-count
  natural-death-count
  NN_Deaths 
  Births
  hunter-kill-count
  bait-kill-count
  %-females-sterile
  week
  year
  %                    ; used to display sum of attractiveness inputs in user interface, to help ensure factors add up to 100%
  end-run
  
  ; program function
  Avg_Sounder_Size
  bait-timer           ; used to place bait at user-specified intervals
  kills-this-week      ; used in hunting function
  sounderless-females  ; used to regroup females
  first-member         ; "
  next-member          ; "
  
  ; added for BehSpc run
;  #_Hunters
;  Base_Kill_Rate
  
;  #_Baits
  #_Baits/Pig_for_100%_Effect
;  Place_Bait_Every___Weeks
;  #_Patches_to_Bait

  min-x
  max-x
  min-y
  max-y
]

pigs-own
[
  age               ; in weeks 
  own-sounder       ; which sounder it belongs to
  travel-this-tick
  sterility-counter
]

sounders-own
[
  piglist           ; list of all the pigs in that sounder
  stock             ; how many pigs the sounder comprises
  travel-this-tick
  farrow-week       ; which weeks (semi-anmnually) pigs in that sounder give birth
]

patches-own
[
  tmp                    ; used to import map values during Setup
  pig-access             ; 0 = no, othewise yes
  people-access          ; "
  baited?                ; "
  dist-to-water          ; range 0 to 35  
  dist-to-water-wet      ; distance from center of patch to water during wet season.  scaled to range 0 to 100.  
  dist-to-water-dry      ; distance from center of patch to water during dry season.  "
  mast-potential         ; based on vegetation and elevation relative to local area.  range 0 to 100
  mast                   ; actual mast content; varies seasonally
  patch-attractiveness   ; combines water, mast, and random attractiveness values, each weighted per user input
  resident_sounder       ; whose "pig sign" is detectable here, if anyone's; whose range this is
  resident_boar          ; "
  sounder-sign-strength  ; detectable "pig sign" fades over a user-specified period.
  boar-sign-strength     ; "
  range-repulsion        ; scaled representation of other "pig sign" strength
  attractiveness         ; combines patch-attractiveness with range-repulsion:  used by each pig when deciding direction of travel
]

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to SETUP
  reset-timer
  ; clear world
  clear-turtles
  clear-all-plots
  clear-patches
  reset-ticks                        ; set time 0

  ; reset counters/output
  set bait-timer 0
  set pig-count 0
  set red-count 0
  set yellow-count 0
  set blue-count 0
  set adult-male-count 0
  set juv-male-count 0
  set female-count 0
  set adult-female-count 0
  set juv-female-count 0
  set sounder-count 0
  set natural-death-count 0
  set NN_Deaths 0
  set Births 0
  set hunter-kill-count 0
  set bait-kill-count 0
  set % %H2O + %Mast + %Random
  set week 0
  set year 1
  set end-run 0
  
  set #_Baits/Pig_for_100%_Effect 2

  set min-x 0
  set max-x 0
  set min-y 0
  set max-y 0

  set-default-shape pigs "feralpig"
  set Avg_Sounder_Size 2.59          ; data specific to Fort Benning (Hanson 2006)

  ask patches
  [ 
    set resident_sounder -1      ; patch is not part of any sounder's range.  when patch is claimed (visited) the -1 will be replaced by that sounder's unique index #
    set resident_boar -1
    set sounder-sign-strength 0
    set boar-sign-strength 0
  ]
  
  read-map "maps/people_access"                               ; excludes hunting from restricted areas
  ask patches [ set people-access tmp ]
  read-map "maps/pig_initial"                                 ; excludes pigs from urban areas and open water.  low-res map for initial population distribution
  ask patches [ set pig-access tmp ]
  ask patches [ if pig-access = -9999 [ set pig-access 0 ] ]
  read-map "maps/cost_to_water"                               ; using for wet season
  ask patches [ set dist-to-water-wet 3.125 * tmp ]           ; scales to a 0 to 100 range
  ask patches [ set dist-to-water-dry 3.125 * tmp ]           ; scales to a 0 to 100 range
  ask patches [ set dist-to-water dist-to-water-dry ]         ; model starts in dry season
  read-map "maps/mast_pct"
  ask patches [ set mast-potential tmp ]
  ask patches [ set mast mast-potential ]                     ; model starts in mast season
  
  ; create females
  create-pigs Initial_Population / 2  ; sex ratio @ Fort Benning is 1:1 (Hanson 2006)
  [
    setxy random-xcor random-ycor
    while [ [ pig-access ] of patch-here = 0 or count other pigs-here > 1] [ setxy random-xcor random-ycor ]  ; moves pigs out of exclusion areas and prevents range overlap
    set color red
    set size 3
    set own-sounder nobody
    set sterility-counter 0
    set age random ( 676 - 52 )       
    set age (age + 43)               ; set age between 43 weeks (youngest that can farrow in model) and 13 years (maximum age per literature).
  ]
  
  ; assign females to sounders
  set sounderless-females pigs with [ own-sounder = nobody and color != blue ]
  while [ count sounderless-females > 0 ]
  [ set first-member one-of sounderless-females
    create-sounder
    set sounderless-females pigs with [ own-sounder = nobody and color != blue ]
  ]

  ; create adult male pigs (boars)
  create-pigs Initial_Population / 2  ; 1:1 sex ratio (Hanson 2007)
  [
    setxy random-xcor random-ycor
    while [ [ pig-access ] of patch-here = 0 ] [ setxy random-xcor random-ycor ]  ; moves boars out of exclusion areas
    set color blue 
    set size 3
    set own-sounder nobody         ; boars travel solo and don't have sounders
    set age random ( 676 - 32 )     
    set age (age + 32)             ; set age between 32 weeks (min. eligible to breed) and 13 years (max).
  ]

  read-map "maps/pig_access"                                                  ; a higher-resolution map of pig-exclusion areas, used in initial placement of pigs
  ask patches [ set pig-access tmp ]
  ask patches [ if pig-access = 0  [ set pig-access 1 ] ]
  ask patches [ if pig-access > 35 [ set pig-access 0 ] ]
  ask patches [ if pig-access = 0  [ set pcolor grey ] ]
  ask patches [ if pig-access = 0 and dist-to-water = 0 [ set pcolor 103 ] ]

  check_%
  update_monitors
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to read-map [ map-name ]
  file-open map-name
  let line file-read-characters 6
  let map-cols file-read   print map-cols
  set line file-read-characters 6
  let map-rows file-read   print map-rows          
  set line file-read-line  print line
  set line file-read-line  print line
  set line file-read-line  print line
  set line file-read-line  print line
  let map-row 0                                    
  let row max-pycor                                 ; start import of patch data at top row
  let val 0
  while [ map-row < map-rows ]
  [ let map-col 0
    let col min-pxcor                               ; start import of patch data at left column
    while [ map-col < map-cols ]
    [ set val file-read
      if (row >= min-pycor) AND (col <= max-pxcor)
      [ ask patch col row [ set tmp val ]
      ]
      set map-col map-col + 1
      set col col + 1
    ]
    set map-row map-row + 1
    set row row - 1
  ]
  file-close
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to GO 
  check_%
  advance_world
  update-patch-values
  go_sounders
  go_pigs
  hunt
  bait
  update_monitors
  tick
;  if pig-count > Stop_Above___Pigs [ stop ]
;  [ print "Pig population has exceded user-specified limit"; " ; run stopped."
;    stop 
;  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 

to Ramp&Run

;Bait 14
;Hunt 0 kills/week  500 baits; 250 patches; q 4 weeks


  if week = 0 and year = 1
  [
    ; set bait parameters
;    set #_Baits 500
;    set #_Baits/Pig_for_100%_Effect 2
;    set Place_Bait_Every___Weeks 4
;    set #_Patches_to_Bait 250
;    set min-x 0
;    set max-x 50
;    set min-y 0
;    set max-y 50
    
    ; set hunt parameters
;    set #_Hunters 30
;    set Base_Kill_Rate 1
      
    ; disable bait and hunt for rampup
;    set Hunt_Mode "Off"
;    set Bait_Mode "Off"
  ]
  
  if year > 2                             ; enable bait/hunt for run
  [
;    if Hunt? = "yes" [ set Hunt_Mode "Non-dependent" ]         ; "Non-dependent"  "Density-based"
;    if Hunt? = "no" [ set Hunt_Mode "Off" ]
;    if Bait? = "yes" [ set Bait_Mode "Random Contraceptive" ]  ; "Block Contraceptive"  "Random Contraceptive"  "Block Lethal"  "Random Lethal"
;    if Bait? = "no" [ set Bait_Mode "Off" ]
  ]


  ifelse year < 13                            ; 2-year rampup + 10-year run
  [ check_%
    advance_world
    update-patch-values
    go_sounders
    go_pigs
    if year > 2 
    [ hunt
      bait
    ]
    update_monitors
    tick
  ]
  [ stop
  ]
  if pig-count > 12000 [ stop ]
end
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to update-patch-values
  ask patches 
  [ 
    ; apply seasonal changes in surface water and mast
    if week = 8 [ set dist-to-water dist-to-water-dry ]                       ; week 1 is 1st week of Jan., dry season is March - August
    if week = 34 [ set dist-to-water dist-to-water-wet ]                      ; winter/wet season is Sept - Feb
;    if week = 39 [ set mast ( .75 * mast-potenti al ) + random ( mast-potential - ( .75 * mast-potential ) ) ]  ;  week 1 is 1st week of Jan., mast season starts Sept (?).  adds 25% variability to mast crop
    if week = 39 [ set mast mast-potential ]                                  ; week 1 is 1st week of Jan., mast season starts Sept (?)
;    if week = 1 [ set mast mast-potential / 2 ]                               ; assume mast dwindling, halved by Jan 1
    if week = 13 [ set mast 0 ]                                               ; assume mast gone by April 1
; note:  water supercedes mast in coloring map:  when transition from wet to dry season, mast-containing patches that dry out update to brown

    ; apply user-specified weighting to attractiveness values
    let water-weight dist-to-water * %H2O / 100                               ; applies user % for how important water is in pig travel
    let mast-weight mast * %Mast / 100                                        ; (as for water-weight)
    set patch-attractiveness ( water-weight + mast-weight + random %Random )  ; adds random weighting for presumed unknown attractiveness factors
    
    ; update ranges (detectable pig sign)
    if sounder-sign-strength > 0 [ set sounder-sign-strength sounder-sign-strength - 1 ]
    if sounder-sign-strength = 0 [ set resident_sounder -1 ]
    if boar-sign-strength > 0 [ set boar-sign-strength boar-sign-strength - 1 ]
    if boar-sign-strength = 0 [ set resident_boar -1 ]

    ; remove bait if present
    if baited? != 0
    [ set baited? 0
      set pcolor black
    ]
 
    ; update map color display
    ifelse pig-access = 0  
    [ set pcolor grey 
      if dist-to-water = 0 [ set pcolor 103 ]
    ]
    [ set pcolor black
      if dist-to-water > 0 
      [ 
        if mast > 15 [ set pcolor 32 ]                    ; darker brown indicates lighter mast
        if mast > 35 [ set pcolor 34 ]                    ; lighter brown indicates moderate to heavier mast
      ]
      if resident_sounder != -1 
      [ if show-ranges = "Sounder_Sign-Strength" [ set pcolor 10 * sounder-sign-strength / Pig_Sign_Duration + 130 ]
        if show-ranges = "Sounder_ID" [ if resident_sounder > -1 [ set pcolor resident_sounder * 10 + 5 ] ]
      ]
      if resident_boar != -1
      [ if show-ranges = "Boar_Sign-Strength" [ set pcolor 10 * boar-sign-strength / Pig_Sign_Duration + 80 ] 
        if show-ranges = "Boar_ID" [ if resident_boar > -1 [ set pcolor resident_boar * 10 + 5 ] ]
      ]
    ]
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to go_sounders
  ask sounders
  [
    if ( piglist = [] ) [ die ]
    set piglist remove-duplicates piglist
    move_sounders
    farrow
  ]  
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to fledge          ; most mature females and all mature males leave the sounder.  gilts (young adult females) form new sounders, while boars (males) go solo
  fledge_males
  fledge_females
  regroup_females
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to fledge_males
  ask pigs
  [ if color = blue and age > 31 and own-sounder != nobody               ; boars leave sounder at 8 months
    [ ask own-sounder [ set piglist remove [who] of myself piglist ]
      set own-sounder nobody
      let possible-ranges patches with [pig-access != 0 and count neighbors with [ pig-access != 0 ] > 7 and resident_boar = -1 and count neighbors with [ resident_boar = -1 ] > 7 ]
      set possible-ranges possible-ranges with [ distance myself > 40 ]  ; boars establish territories some distance from their birth sounder
      let my-new-range one-of possible-ranges
      if my-new-range != nobody 
      [ move-to my-new-range
      ]
    ]
  ]
end

; The "fledge" procedures move each pig to an unclaimed area, even if the linear path to that area crosses patches that pigs are unlikely to traverse (eg urban areas, open water).
; For a more accurate model, this procedure could be amended to weight these physical barriers.
  
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to fledge_females                                                                    ; this section kicks all but the oldest 2 or 3 females out of the sounder
  ask sounders
  [ ;let stock# random 100                                                             ; decide how many adult females the sounder will comprise (avg. based on Fort Benning data)
    ;if stock# < 59 [ set stock 3 ]
    ;if stock# >= 59 [ set stock 2 ]
    let adultfemale []
    foreach piglist
    [ if [ age ] of pig ? > 26 and [color] of pig ? != blue [ set adultfemale lput ? adultfemale ]  ;  create a list of all adult females in sounder...
    ]  
    set adultfemale sort adultfemale                                                 ; sort them by age (index)
    while [ length adultfemale > stock ]                                             ; as long as there are more than the chosen # of females in the sounder...
    [ ask pig last adultfemale [ set own-sounder nobody ]                            ; kick the youngest adult female out
      set piglist remove-item position last adultfemale piglist piglist              ; reset lists
      set adultfemale remove-item position last adultfemale adultfemale adultfemale
    ]
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to regroup_females
  set sounderless-females pigs with [ own-sounder = nobody and color != blue ]
  while [ count sounderless-females > 0 ]
  [ choose-a-range
    create-sounder
    set sounderless-females pigs with [ own-sounder = nobody and color != blue ]
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to choose-a-range
  set sounderless-females pigs with [ own-sounder = nobody and color != blue ]
  ask one-of sounderless-females
    [ set first-member self
      let possible-ranges patches with [ pig-access != 0 and resident_sounder = -1 and count neighbors with [ pig-access != 0 and resident_sounder = -1 ] > 7 ]
      let n count possible-ranges
      if n > 29 [ set possible-ranges min-n-of 30 possible-ranges [ distance myself ] ]       ; relative to boars, gilts establish new ranges close to their birth sounder
      if n < 30 [ set possible-ranges min-n-of n possible-ranges [ distance myself ] ]
;; NOTE:  had trouble with new sounders forming too close together as pop. density increased; n=30 is a way of spreading them out a little more
      let my-new-range one-of possible-ranges
      if my-new-range != nobody 
      [ move-to my-new-range
      ]
    ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

to create-sounder
  create-sounders 1
  [ set size 1
    set piglist []
    set color white
    assign-breeding-date
    hide-turtle
    set piglist lput [ who ] of first-member piglist                                ; start a new sounder by adding an available female to it
    ask first-member
    [ set own-sounder myself
    ]
    set xcor [ xcor ] of first-member                                               ; place the new sounder at her location
    set ycor [ ycor ] of first-member
    let stock# random 100                                                           ; decide how many adult females the sounder will comprise (based on Fort Benning average)
    if stock# < 59 [ set stock 3 ]
    if stock# >= 59 [ set stock 2 ]
    let needed stock - 1
    while [ needed > 0 ]                                                             ; until this sounder is fullly stocked...
    [ set sounderless-females pigs with [ own-sounder = nobody and color != blue ]
      if count sounderless-females > 0                                              ; and as long as there are sounderless females...
      [ set next-member min-one-of sounderless-females [ distance myself ]          ; add the nearest one to this sounder
        set piglist lput [ who ] of next-member piglist
        ask next-member
        [ set own-sounder myself
          move-to patch-here
        ]
      ]
      set needed needed - 1
    ]
    
    set sounderless-females pigs with [ own-sounder = nobody and color != blue ]
    if count sounderless-females = 1                                              ; if there's only one sounderless female left    
    [ set next-member one-of sounderless-females                                  ; add her to this sounder
        set piglist lput [ who ] of next-member piglist
        ask next-member
        [ set own-sounder myself
          move-to patch-here
        ]
    ]
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to go_pigs
    apply_attrition
    age_pigs
    move_pigs
    fledge
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to apply_attrition
  if Attrition                                              ; using Hanson 2006 apparent survival rates, which classify pigs up to 8 months as juveniles and > 8 months as adults
  [ ask pigs                                                ; weekly growth rate = ( 1 + annual growth rate ) ^ ( 1 / 52 ) - 1      
    [ ifelse age > 35                                       ; 35 weeks = 8 months  

      ; adult attrition 
      [ ifelse (color = blue)
        [ if random 10000000000 < 0298350430                      ; adult male annual survival rate 0.207 = 0.793 annual attrition = 0.0298350430 weekly attrition. 
          [ set natural-death-count ( natural-death-count + 1 )
            pig_die 
          ] 
        ]
        [ if random 10000000000 < 0217327534                      ; adult female annual survival rate 0.319 = 0.681 annual attrition = 0.0217327534 weekly attrition.
          [ if (own-sounder != nobody)                            ; if in sounders,
            [ set natural-death-count ( natural-death-count + 1 )
              pig_die 
            ]
          ]
        ]
      ]
    
      ; juvenile attrition
      [ ifelse age < 5
 
        ; birth to one month
        [ let weekly_Neonate_Mortality Neonate_Mortality / 4      ; spread neonate mortality over 4 weeks
          if random 100 < ( weekly_Neonate_Mortality )
          [ set natural-death-count ( natural-death-count + 1 )
            set NN_Deaths NN_Deaths + 1
            pig_die
          ]
        ]
        
        ; > one month to eight months
        [ ifelse color = blue
          [ if random 10000000000 < 0304766588                    ; juvenile male annual survival rate 0.200 = 0.800 annual attrition = 0.0304766588 weekly attrition.
            [ set natural-death-count ( natural-death-count + 1 )
              pig_die
            ]
          ]
          [ if random 10000000000 < 0222104486                    ; juvenile female annual survival rate 0.311 = 0.689 annual attrition = 0.0222104486 weekly attrition.
            [ set natural-death-count ( natural-death-count + 1 )
              pig_die 
            ]
          ]
        ]
      ]
    ]
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to pig_die
  if own-sounder != nobody
    [ ask own-sounder [ set piglist remove [who] of myself piglist]
    ]
  die 
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to age_pigs
  ask pigs
  [
    set age (age + 1)
;    if age > 32 [set size 4]
    if color = yellow
    [ if sterility-counter = 0 [ set color red ]
      set sterility-counter sterility-counter - 1
    ]
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 

to farrow
  if Recruitment
  [ if week = farrow-week or week = farrow-week + 26                                      ; if it's this sounder's time to farrow
    [ let new-moms pigs with [ own-sounder = myself and age > 42 and color != yellow ]
      let new-moms-count count new-moms                                                   ; count fertile females old enough to give birth
      while [ new-moms-count > 0 ]
      [ let first-parity-count count new-moms with [ own-sounder = myself and age < 52 ]
        while [ first-parity-count > 0 ]
        [ hatch-pigs 5 [ hatch-piglet ]                                                   ; gilts (young females) have an average of 5 piglets per litter
          set first-parity-count first-parity-count - 1
          set new-moms-count new-moms-count - 1
        ]
        ifelse random 100 > 87                                                            ; sows one year or older average 6.87 piglets per litter
        [ hatch-pigs 6 [ hatch-piglet ]
        ]
        [ hatch-pigs 7 [ hatch-piglet ]
        ]
        set new-moms-count new-moms-count - 1
      ]
    ]
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to hatch-piglet
  set size 3
  set age 0
  ifelse random 100 > 50 [ set color blue ] [ set color red ]
  set own-sounder myself
  ask own-sounder
    [ set piglist lput [who] of myself piglist ]
  set Births Births + 1
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 
    
to move_pigs  
  ask pigs
  [ ifelse own-sounder = nobody
    [ move_boars ]
    [ set xcor [ xcor ] of own-sounder
      set ycor [ ycor ] of own-sounder  ;  place sows and piglets at their sounders
    ]
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%% 

to move_sounders
  let travel-count 0
  let me who
  set travel-this-tick random 22                                                   ;  sounders move up to 0.61 km/day; up to 4.28 km or 21.4 patch lengths/week
  while [ travel-count < travel-this-tick ]
  [ ask neighbors                                                                  ;  for each of the 8 neighboring patches ...
    [ ifelse resident_sounder != -1 and resident_sounder != me                     ;  if patch has a resident sounder and it's not me ...
      [ set range-repulsion 100 * sounder-sign-strength / Pig_Sign_Duration
      ]
      [ set range-repulsion 0
      ]
      set attractiveness ( - range-repulsion + patch-attractiveness )              ;  decrease my attraction to that patch based on strength of that sounder's pig sign
    ]
    let possibilities neighbors with [ pig-access != 0 and attractiveness > 0 and range-repulsion = 0 ]
    let possibility max-one-of possibilities [ attractiveness ]                    ;  if possible choose the most attractive patch that has no detectable sounder pig sign
    ifelse possibility = nobody
    [ set possibilities neighbors with [ pig-access != 0 and attractiveness > 0 ]
      set possibility min-one-of possibilities [ range-repulsion ]                 ;  otherwise, choose the patch with the least (oldest and most faded) sounder pig sign
    ]
    [ move-to possibility
      ask patch-here
      [ set resident_sounder me                                                    ;  place my pig sign in each patch I visit
        set sounder-sign-strength Pig_Sign_Duration                                ;  pig sign fades linearly over the time period specified by the user
      ]
    ]
    set travel-count travel-count + 1
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  
  
to move_boars
  let travel-count 0
  let me who
  set travel-this-tick random 40                                       ; boars move up to 1.11 km/day = 7.77 km or 38.85 patch lengths/week
  while [ travel-count < travel-this-tick ]
  [ ask neighbors
    [ ifelse resident_boar != -1 and resident_boar != me
      [ set range-repulsion 100 * boar-sign-strength / Pig_Sign_Duration    
      ]
      [ set range-repulsion 0
      ]
      set attractiveness ( - range-repulsion + patch-attractiveness )
    ]
    let possibilities neighbors with [ pig-access != 0 and attractiveness > 0 and range-repulsion = 0 ]
    let possibility max-one-of possibilities [ attractiveness ]
    ifelse possibility = nobody
    [ set possibilities neighbors with [ pig-access != 0 and attractiveness > 0 ]
      set possibility min-one-of possibilities [ range-repulsion ]
    ]
    [ move-to possibility
      ask patch-here
      [ set resident_boar me
        set boar-sign-strength Pig_Sign_Duration
      ]
    ]
    set travel-count travel-count + 1
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  
 
to update_monitors
  set-current-plot "Pig_Population"
  set red-count count pigs with [ color = red ]
  set yellow-count count pigs with [ color = yellow ]
  set pig-count count pigs
  set-current-plot-pen "Females"
  plotxy ticks ( red-count + yellow-count )

  set blue-count count pigs with [ color = blue ]
  set adult-male-count count pigs with [ color = blue and age > 31 ]
  set juv-male-count blue-count - adult-male-count 
  set-current-plot-pen "Males"
  plotxy ticks blue-count
  
  set-current-plot-pen "Total Pigs"
  plotxy ticks count pigs with [ shape = "feralpig" ]

  set female-count count pigs with [ color != blue ]
  set sounder-count count sounders
  set adult-female-count ( count pigs with [ color != blue and age > 26 ] )
  set juv-female-count female-count - adult-female-count
  if female-count > 0 
  [ set %-females-sterile ( 100 * yellow-count / female-count ) 
  ]
  
  set-current-plot "Age_Structure"
  set-current-plot-pen "<1" 
  plotxy ticks count pigs with [ age < 52 ]
  set-current-plot-pen "1"
  plotxy ticks count pigs with [ age >= 52 and age < 104 ]
  set-current-plot-pen "2" 
  plotxy ticks count pigs with [ age >= 104 and age < 156 ]
  set-current-plot-pen "3" 
  plotxy ticks count pigs with [ age >= 156 and age < 208 ]
  set-current-plot-pen ">3" 
  plotxy ticks count pigs with [ age >= 208 ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to advance_world  
  set bait-timer bait-timer - 1
  ifelse week = 52
  [ 
    set week  1
    set year year + 1
  ]
  [ set week week + 1
  ]
end

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to hunt
; reduce attractiveness for recent hunting hx?
  if Hunt? = "yes"
  [ let huntable-pigs count pigs with [ [people-access] of patch-here = 1 ]
;    if Hunt_Mode = "Non-dependent" [ set kills-this-week #_Hunters * Base_Kill_Rate ]
;    if Hunt_Mode = "Density-based" [ set kills-this-week #_Hunters * Base_Kill_Rate * ( ( huntable-pigs / 100000 ) ^ 2 ) ]
;    if Hunt_Mode = "Density-based" [ set kills-this-week ( huntable-pigs / 500 ) ^ 2 ]
    set kills-this-week ( huntable-pigs / 500 ) ^ 2
; future models could include a place-specific density-dependency function
    let #shot 0
    set kills-this-week monthly-hunt-kill * ( 12 / 52 )
    while [ #shot < kills-this-week ]
    [ if not any? pigs with [ [ people-access ] of patch-here = 1 ] [ stop ]
      ask one-of pigs  
      [ if [ people-access ] of patch-here = 1
        [ set hunter-kill-count hunter-kill-count + 1
          set #shot #shot + 1 
          pig_die
        ]
      ]
    ]
  ]
end

; a study of where pigs were killed by hunters might help optimize hunting effectiveness
; what really happens when a hunter encounters a sounder?  should a differential kill rate be applied to boars vs sounders?
; variability of hunter access (based on terrain, distance to roads, knowledge of where pigs are or previous hunting success) is not modeled here.

;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to bait
  if Bait? = "yes"
  [ if bait-timer < 1                                                                                           ; if it's time to bait do the following

    ; bait all indicated patches
    [ if Bait_Mode = "Block Contraceptive" or Bait_Mode = "Block Lethal"
      [ ask patches
        [ if ( pxcor > min-x) and ( pxcor < max-x ) and ( pycor > min-y ) and ( pycor < max-y ) 
          [ set baited? 1
            set pcolor violet
          ]
        ]
      ]
      if Bait_Mode = "Random Contraceptive" or Bait_Mode = "Random Lethal"
;      [ let patches-to-bait #_Patches_to_Bait
      [ let patches-to-bait #_Baits / baits-per-patch
        while [ patches-to-bait > 0 ]
        [ let patch-to-bait one-of patches with [ pig-access = 1 ]
          ask patch-to-bait [ set baited?  1
          set pcolor violet ]
          set patches-to-bait patches-to-bait - 1
        ]
      ]
    ]
    
    ; calculate bait density
    let #baited-patches count patches with [baited? = 1]
    if #baited-patches != 0 
    [ ; let baits-per-patch #_Baits / #baited-patches

      ask patches
      [ if baited? = 1 and count pigs-here > 0                                                                  ; for each baited patch that contains pigs...
        [ let odds-against-eating-bait ( #_Baits/Pig_for_100%_Effect / ( baits-per-patch / count pigs-here ) )  ; calculate odds that each pig will eat bait...
; 1-in-2 odds here are arbitrary; overall baiting effectiveness can be adjusted in several places in this model
          ask pigs-here
          [ let ate-bait? random odds-against-eating-bait                                                       ; then roll dice for each pig to decide whether it eats bait
            if ate-bait? < 1
            [ if Bait_Mode = "Block Lethal" or Bait_Mode = "Random Lethal"                                      ; if any pig ate lethal bait, kill that pig
              [ set bait-kill-count bait-kill-count + 1
                pig_die
              ]
              if Bait_Mode = "Block Contraceptive" or Bait_Mode = "Random Contraceptive"
;              [ if color = red [ set color yellow ]                                                             ; if female pig ate contraceptive bait, make pig sterile
              [ if color = red [ set color yellow set sterility-counter #weeks_sterile ]                      ; if female pig ate contraceptive bait, make pig sterile for x# weeks
              ]
            ]
          ]
        ]
      ]
    set bait-timer Place_Bait_Every___Weeks
    ]
  ]
end

; incorp bait into color update - create a "just baited" or clear bait beginning of next step
;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to assign-breeding-date  ; all adult females will farrow twice yearly, with sounder farrowing synchronized
  Let breeding-date-odds random 3 ; 1 in 3 odds of farrowing off-peak (Jolley's top 6 farrowing months accounted for 67.5% of farrowing)
  ifelse breeding-date-odds = 0
  [ set farrow-week random 26
  ]
  [ let which-peak random 2
    if which-peak = 0 [ set farrow-week 9 + random 4 ]    ; these pigs will farrow in the spring (March) - season is 4 weeks long
    if which-peak = 1 [ set farrow-week 27 + random 10 - 26 ]  ; these pigs will farrow in the fall (July through Sept) - season is 10 weeks long
  ]
end


;%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%  

to check_%
  set % ( %H2O + %Mast + %Random )
  while [ % != 100 ] 
  [ sound:play-drum "Cowbell" 64
; why won't beep work?
    output-print "must total 100%"
    wait 3
    set % ( %H2O + %Mast + %Random )
  ]
end

to go10
let rem_steps 520
while [ rem_steps > 0 ]
[ go
  set rem_steps rem_steps - 1
;  if pig-count > Stop_Above___Pigs [ stop ]
]
end

to rampup
let rem_steps 104
while [ rem_steps > 0 ]
[ go
  set rem_steps rem_steps - 1
]
end
@#$#@#$#@
GRAPHICS-WINDOW
220
17
1020
838
100
100
3.93035
1
10
1
1
1
0
1
1
1
-100
100
-100
100
1
1
1
Week
30.0

BUTTON
5
54
78
87
Set Up
SETUP
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
81
54
152
87
Go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
5
19
211
52
Initial_Population
Initial_Population
6
5000
2000
1
1
pigs
HORIZONTAL

MONITOR
1027
494
1085
539
Females
female-count
0
1
11

MONITOR
1027
445
1084
490
Males
blue-count
0
1
11

SWITCH
4
123
96
156
Attrition
Attrition
0
1
-1000

PLOT
1024
10
1224
160
Pig_Population
Time
Number of pigs
0.0
0.0
0.0
200.0
true
true
"" ""
PENS
"Total Pigs" 1.0 0 -16777216 true "" ""
"Females" 1.0 0 -2674135 true "" ""
"Males" 1.0 0 -13345367 true "" ""

SWITCH
98
123
211
156
Recruitment
Recruitment
0
1
-1000

MONITOR
1026
396
1084
441
Pigs
pig-count
0
1
11

MONITOR
1026
658
1089
703
Hunt Kills
Hunter-kill-count
0
1
11

MONITOR
1026
727
1108
772
Sterile Females
yellow-count
0
1
11

MONITOR
1093
658
1156
703
Bait Kills
bait-kill-count
0
1
11

MONITOR
1111
727
1207
772
% Sterile Females
%-females-sterile
0
1
11

INPUTBOX
6
397
140
457
Pig_Sign_Duration
52
1
0
Number

MONITOR
1026
336
1076
381
Year
year
0
1
11

CHOOSER
7
339
110
384
Show-Ranges
Show-Ranges
"None" "Sounder_Sign-Strength" "Sounder_ID" "Boar_Sign-Strength" "Boar_ID"
0

MONITOR
1080
336
1130
381
Week
week
0
1
11

MONITOR
1087
397
1173
442
Baseline Deaths
natural-death-count
0
1
11

BUTTON
155
55
210
88
Go step
Go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
1027
543
1085
588
Sounders
sounder-count
0
1
11

INPUTBOX
6
263
63
323
%H2O
30
1
0
Number

INPUTBOX
65
263
122
323
%Mast
30
1
0
Number

INPUTBOX
124
263
184
323
%Random
40
1
0
Number

MONITOR
6
214
184
259
Attractiveness Weighting (%)
%
0
1
11

MONITOR
1089
494
1150
539
Adult Fem.
adult-female-count
0
1
11

MONITOR
1154
494
1220
539
Juv. Fem.
juv-female-count
0
1
11

MONITOR
1153
445
1219
490
Juv. Males
juv-male-count
0
1
11

MONITOR
1088
445
1149
490
Ad. Males
adult-male-count
0
1
11

SLIDER
4
158
212
191
Neonate_Mortality
Neonate_Mortality
0
100
42
1
1
%
HORIZONTAL

MONITOR
1026
611
1083
656
Births
Births
17
1
11

MONITOR
1087
611
1174
656
Neonate Deaths
NN_Deaths
0
1
11

PLOT
1026
165
1226
315
Age_Structure
Time
Number of Pigs
0.0
10.0
0.0
10.0
true
true
"" ""
PENS
"<1" 1.0 0 -2674135 true "" ""
"1" 1.0 0 -955883 true "" ""
"2" 1.0 0 -1184463 true "" ""
"3" 1.0 0 -10899396 true "" ""
">3" 1.0 0 -13345367 true "" ""

BUTTON
7
761
79
794
Ramp-Up
rampup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
82
762
168
795
10-Year Run
go10
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
7
796
137
829
Ramp & Run (B.S.)
Ramp&Run
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

MONITOR
1027
774
1185
819
#_Baits/Pig_for_100%_Effect
#_Baits/Pig_for_100%_Effect
17
1
11

INPUTBOX
5
613
66
673
#_Baits
2500
1
0
Number

INPUTBOX
101
475
214
535
monthly-hunt-kill
0
1
0
Number

INPUTBOX
69
613
215
673
Place_Bait_Every___Weeks
4
1
0
Number

INPUTBOX
103
551
195
611
baits-per-patch
2
1
0
Number

CHOOSER
6
476
98
521
Hunt?
Hunt?
"yes" "no"
0

CHOOSER
7
566
99
611
Bait?
Bait?
"yes" "no"
0

CHOOSER
100
676
217
721
Bait_Mode
Bait_Mode
"Block Contraceptive" "Random Contraceptive" "Block Lethal" "Random Lethal"
1

CHOOSER
5
675
97
720
#weeks_sterile
#weeks_sterile
13 26 39 52 104
0

@#$#@#$#@
## WHAT IS IT?

This section could give a general understanding of what the model is trying to show or explain.

## HOW IT WORKS

Setup

Program Flow:  
Go
	Check %
	Advance world
	Update patch values
	Go sounders
		Move sounders
		Farrow (recruitment)
	Go pigs
		Apply attrition
		Age pigs
		Move pigs
		Fledge
			Fledge males
			Fledge females
			Regroup females
				Choose a range
				Create sounder
	Hunt
	Bait
	Update monitors
	Tick


## USING THE MODEL

-- Tan boxes are output - information that the model provides to you. --

The top boxes to the right of the map track the model calendar, while the "Week" at the top of the map shows total weeks elapsed.

Further right, counters show the total number of pigs, total males and females, adults and juveniles, and the number of sounders.

The Pig Population graph shows how the population changes over time.

Below the calendar boxes are monitors that track the total number of pigs that have died as a result of control efforts or other causes, and showing number and percent of females that have been sterilized by contraceptive bait.

-- You can change model parameters using the blue buttons and boxes. --

All of these can be changed while the model is running or stopped, before or after setup.  If you type into a box, you must click in another box before the model will recognize your input.

Setup:  Initialize world, import patch data (maps), and create initial pig population
	Initial population:  Indicate number of pigs to create in setup.  Changes to this box take effect the next time you click Setup.

Go:  Start and stop the model.  
Go Step:  Advance the model one week.

Attrition:  Turn mortality function on and off.  
Recruitment:  Turn farrowing function on and off.
	Neonate Mortality:  Specify mortality rate for piglets from birth to one month.  Default .25

Attractiveness Weighting:  Assign relative importance to water, mast, and other "random" factors in feral pigs' decisions about where to move.  Values should total 100%.

Show Ranges:  Delineate individual boar or sounder ranges, or see how frequently pigs are visiting all parts of their ranges.

Pig Sign Lasts ___ Weeks:  Indicate how long evidence of a boar or sounder visit to a patch is detectable by other pigs, identifying that patch as someone's range.  Default 52

Hunt Mode:  Decide whether rate of pig removal by hunting will depend on the population density.  Consider whether the hunting program will be managed to encourage hunting as the swine population declines.
	# of Hunters:  Indicate how many people hunt each week.
	Base Kill Rate:  Designate success rate for hunters.  This is the actual number of pigs killed per hunter for non-dependent mode.  When hunting success is density-based, base kill rate is weighted against the number of pigs in accessible areas.

Bait Mode:  Choose between lethal and contraceptive baits.
	# of Baits:  Indicate the total number of baits to be placed during each scheduled baiting.
	# Baits/Pig for 100% Effect:  Indicate how many baits would have to be available for (not necessarily consumed by) every pig in order to ensure that every pig consumes enough to be affected.  Consider factors such as minimum effective dose, likelihood of finding the bait, competition between pigs, and consumption by other wildlife.
	Min and Max x and y:  Delineate the area in which bait will be placed.  Values for each axis range from -200 to 200.  A difference of 1 unit (i.e. min-x = 3, max-x - 4) corresponds to approximately 176 meters, or 0.11 mile.
	Place Bait Every ___ Weeks:  Establish a schedule for bait placement (default 4)

Speed and visual update controls are located in the menu above the map display.  Use these to speed up processing as the population grows, or slow it down if the map display doesn't update on every time step.

## THINGS TO NOTICE

## THINGS TO TRY

## EXTENDING THE MODEL

Things to add or change in the procedures tab to make the model more complicated, detailed, accurate, etc.

Bait efficacy, duration of sterilizing effect, influence of stations on patch attractiveness  
Bait placement:  multiple locations, accept GPS coord's, lat/long input

## NETLOGO FEATURES

This section could point out any especially interesting or unusual features of NetLogo that the model makes use of, particularly in the Procedures tab.  It might also point out places where workarounds were needed because of missing features.

## RELATED MODELS

This section could give the names of models in the NetLogo Models Library or elsewhere which are of related interest.

## CREDITS AND REFERENCES

Jim Westervelt  
Jennifer L. Burton

Comments or questions?  Contact Jen Burton:  jlburton@.illinois.edu
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

feralpig
false
0
Polygon -7500403 true true 210 210 195 240 180 240 180 210 150 225 120 225 90 210 75 210 72 211 49 209 45 210 15 180 15 135 15 120 45 90 120 105 135 105 165 90 225 120 255 150 300 165 285 180 270 195 240 195 225 195
Polygon -7500403 true true 90 210 75 240 60 240 60 210
Polygon -7500403 true true 195 105 225 75 210 120
Polygon -7500403 true true 30 120 15 105 15 90 45 75 30 90 30 105
Polygon -1 true false 210 180 225 180 240 195 255 210 270 210 255 195 240 180 225 180

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0RC3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="All Trials 2009-08-24" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>Ramp&amp;Run</go>
    <exitCondition>count pigs &gt; 12000</exitCondition>
    <metric>count turtles</metric>
    <metric>hunter-kill-count</metric>
    <metric>yellow-count</metric>
    <metric>timer</metric>
    <enumeratedValueSet variable="Hunt?">
      <value value="&quot;yes&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="monthly-hunt-kill">
      <value value="0"/>
      <value value="25"/>
      <value value="50"/>
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bait?">
      <value value="&quot;yes&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="#_Baits">
      <value value="0"/>
      <value value="2500"/>
      <value value="5000"/>
      <value value="7500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="baits-per-patch">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Place_Bait_Every___Weeks">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Attrition">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Neonate_Mortality">
      <value value="42"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Recruitment">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show-Ranges">
      <value value="&quot;None&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pig_Sign_Lasts___Weeks">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial_Population">
      <value value="2000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%H2O">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%Mast">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%Random">
      <value value="40"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="H75 B5000" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>Ramp&amp;Run</go>
    <exitCondition>count pigs &gt; 12000</exitCondition>
    <metric>count turtles</metric>
    <metric>hunter-kill-count</metric>
    <metric>yellow-count</metric>
    <metric>timer</metric>
    <enumeratedValueSet variable="Hunt?">
      <value value="&quot;yes&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="monthly-hunt-kill">
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bait?">
      <value value="&quot;yes&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="#_Baits">
      <value value="5000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="baits-per-patch">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Place_Bait_Every___Weeks">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Attrition">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Neonate_Mortality">
      <value value="42"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Recruitment">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show-Ranges">
      <value value="&quot;None&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pig_Sign_Lasts___Weeks">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial_Population">
      <value value="2000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%H2O">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%Mast">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%Random">
      <value value="40"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="2010-08-06_Finite_bait" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>Ramp&amp;Run</go>
    <exitCondition>count pigs &gt; 12000</exitCondition>
    <metric>count turtles</metric>
    <metric>hunter-kill-count</metric>
    <metric>yellow-count</metric>
    <metric>timer</metric>
    <enumeratedValueSet variable="Hunt?">
      <value value="&quot;yes&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="monthly-hunt-kill">
      <value value="0"/>
      <value value="25"/>
      <value value="50"/>
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bait?">
      <value value="&quot;yes&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="#_Baits">
      <value value="2500"/>
      <value value="5000"/>
      <value value="7500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="baits-per-patch">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Place_Bait_Every___Weeks">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="#weeks_sterile">
      <value value="13"/>
      <value value="26"/>
      <value value="39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Attrition">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Neonate_Mortality">
      <value value="42"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Recruitment">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show-Ranges">
      <value value="&quot;None&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pig_Sign_Lasts___Weeks">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial_Population">
      <value value="2000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%H2O">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%Mast">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%Random">
      <value value="40"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="2011-02-17_Finite_bait-12&amp;24mo" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>Ramp&amp;Run</go>
    <exitCondition>count pigs &gt; 12000</exitCondition>
    <metric>count turtles</metric>
    <metric>hunter-kill-count</metric>
    <metric>yellow-count</metric>
    <metric>timer</metric>
    <enumeratedValueSet variable="Hunt?">
      <value value="&quot;yes&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="monthly-hunt-kill">
      <value value="0"/>
      <value value="25"/>
      <value value="50"/>
      <value value="75"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bait?">
      <value value="&quot;yes&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="#_Baits">
      <value value="2500"/>
      <value value="5000"/>
      <value value="7500"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="baits-per-patch">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Place_Bait_Every___Weeks">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="#weeks_sterile">
      <value value="52"/>
      <value value="104"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Attrition">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Neonate_Mortality">
      <value value="42"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Recruitment">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show-Ranges">
      <value value="&quot;None&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pig_Sign_Lasts___Weeks">
      <value value="52"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial_Population">
      <value value="2000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%H2O">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%Mast">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="%Random">
      <value value="40"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
